/*****************************************************************************
**+------------------------------------------------------------------------+**
**|                                                                        |**
**|                Copyright 2010 Mistral Solutions Pvt Ltd.               |**
**|                                                                        |**
**|                                                                        |**
**|                                                                        |**   
**| This program is free software; you can redistribute it and/or          |**
**| modify it under the terms of the GNU General Public License as         |**
**| published by the Free Software Foundation; either version 2 of         |**
**| the License, or (at your option) any later version.                    |**
**|                                                                        |**
**| This program is distributed in the hope that it will be useful,        |**
**| but WITHOUT ANY WARRANTY; without even the implied warranty of         |**
**| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the           |**
**| GNU General Public License for more details.                           |**
**|                                                                        |**      
**| You should have received a copy of the GNU General Public License      |**
**| along with this program; if not, write to the Free Software            |**
**| Foundation, Inc., 59 Temple Place, Suite 330, Boston,                  |**
**| MA 02111-1307 USA                                                      |**
**+------------------------------------------------------------------------+**
*****************************************************************************/    

/**
 * \file    spiFlash.c
 *
 * \brief   Support routines for testing SPI-NAND Flash on EVM
 *
 * This file contains the Support routines for testing SPI-NAND Flash on EVM.
 *
 * \author  0043
 *
 * \version 01a,13sep,2010 Created
 */

/*
 *====================
 * Includes
 *====================
 */
#include "DM814x_EVM.h"
#include "spiflash.h"
#include "stdio.h"
/*
 *====================
 * Satic declarations
 *====================
 */
static UINT8 spiflashbuf[SPIFLASH_PAGESIZE + 5];
static UINT8 statusbuf[8];

/*
 *====================
 * Function Implementation
 *====================
 */
 /** **************************************************************************
 * \n \brief Routine to intialize SPI Flash
 *
 * This routine initializes SPI flash by configuring McSPI controller.
 * 
 *
 */
void spiflash_init( )
{
    /* Reset SPI */
    MCSPI_SYSCONFIG |= 0x02; // Bit is automatically reset by hardware
    /* Wait for reset to complete */
    while(MCSPI_SYSSTATUS & 0x01 == 0x00);

    /* Configure MCSPI Module */
    MCSPI_MODULCTRL = 0
	| ( 0 << 8 ) // Data managed by MCSPI_TX(i) and MCSPI_RX(i) registers
	| ( 0 << 7 ) // Multiple word access disabled
	| ( 0 << 4 ) // No delay for first spi transfer ????????????????????????????????
	| ( 0 << 3 ) // Functional mode
	| ( 0 << 2 ) // Master
	| ( 0 << 1 ) // SPIEN is used as a chip select
	| ( 1 << 0 );// Only one channel will be used in master mode

    /* Configure MCSPI Channel 0 */
    MCSPI_CH0CONF = 0
	| ( 0 << 29 ) // Clock divider granularity of power of two
	| ( 1 << 28 ) // The buffer is used to receive data
	| ( 1 << 27 ) // The buffer is used to transmit data
	| ( 1 << 25 ) // 1.5 cycles between CS toggling and first or last edge of SPI clock
	| ( 0 << 24 ) // Start bit polarity
	| ( 0 << 23 ) // Disable start bit
	| ( 0 << 20 ) // SPIEN active between SPI words
	| ( 0 << 19 ) // Turbo is deactivated 
	| ( 0 << 18 ) // Data Line0 selected for reception
	| ( 1 << 17 ) // Data Line1 selected for transmission 
	| ( 0 << 16 ) // No transmission on Data Line0 
	| ( 0 << 15 ) // DMA Read Request disabled 
	| ( 0 << 14 ) // DMA Write Request disabled 
	| ( 0 << 12 ) // Transmit and Receive mode 
	| ( 7 <<  7 ) // SPI word length = 8
	| ( 1 <<  6 ) // SPIEN is held high during the active state
	| ( 8 <<  2 ) // CLKD = 8 Clock devider (Recalibrate according to pll)????????????????????
	| ( 0 <<  1 ) // SPICLK is held high during the active state 
	| ( 0 <<  0 );// Data are latched on even numbered edges of SPICLK

    /* Enable MCSPI channel */
    MCSPI_MODULCTRL = 0x01; // Enable Channel
}
/** **************************************************************************
 * \n \brief Routine to do data transfer
 *
 * This routine transfers the  buffer byte by byte to spiflash and 
 * data from flash is placed in the same buffer.It takes buffer as parameter 
 * which has to be send to the flash along with that length is send as another
 * parameter.
 *
 * \param  *buf     [IN]  Buffer pointer to transfer data
 * \param len       [IN]  Length of the buffer
 *
 *             
 */
void spiflash_cycle( UINT8 *buf, UINT16 len )
{
    UINT16 i;
    
    /* Enable Channel */
    MCSPI_CH0CTRL = 0x01;

    /* SPIFLASH access cycle */
    MCSPI_CH0CONF |= 0x00100000;
    for ( i = 0 ; i <= len ; i++ )
    {
        /* Wait for transmit empty */
        while ( (MCSPI_CH0STAT & 0x02) == 0 );
        
	MCSPI_TX0 = buf[i];  // Write to RX buffer 0

        /* Wait for receive data full */
        while ( (MCSPI_CH0STAT & 0x01) == 0 );
        
	buf[i] = MCSPI_RX0;  // Read from RX buffer 0
        
    }
    
    MCSPI_CH0CONF &= ~0x00100000;

    /* Disable Channel */
    MCSPI_CH0CTRL = 0x00;
}
/** **************************************************************************
 * \n \brief Routine to read the of SPI flash status register
 *
 * This routine sends the read status command to spiflash and reads the status
 * and returns the same.
 *
 * \return
 * \n  returns the status of the register.
 */
UINT8 spiflash_status( )
{
    /* Issue read status command */
    statusbuf[0] = SPIFLASH_CMD_RDSR;
    statusbuf[1] = 0;

    spiflash_cycle( statusbuf, 2 );

    return statusbuf[1];
}
/** **************************************************************************
 * \n \brief Routine to read the Manufacturer ID and device ID of Spiflash 
 *
 * This routine sends the readid command to flash and reads both Ids and 
 * verifies the Manufacturer ID.
 *
 * \return
 *
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 *
 */

STATUS spiFlash_readID ()
{
	STATUS u32RetVal = SUCCESS;

	statusbuf[0] = SPIFLASH_CMD_READID;
	statusbuf[1] = 0x00;
	
	spiflash_cycle( statusbuf, 10 );

	/* The Manufacturrerer ID and the Device ID is read into offset 5 & 6 */
	if (SPIFLASH_MANUF_ID != statusbuf[5])
	{
		u32RetVal = FAILED;
		printf ("Unknown manufacturer ID read: 0x%x.\r\n",statusbuf[5]); 
	} 
	else
	{
		printf ("Manufacturer ID read is 0x%x.\r\n", statusbuf[5]);
		printf ("Device ID read is 0x%x.\r\n", statusbuf[6]); 
	}
	
	return (u32RetVal);
}
/** **************************************************************************
 * \n \brief Routine to read from Spiflash device
 *
 * This routine reads the data from a page .It takes pagenum as a parameter 
 * along with that buffer where read data is stored and length of the buffer. 
 * 
 *
 * \param  src     [IN]     source from where data has to be read
 * \param  dst     [IN]     Buffer where read data has to be stored
 * \param  length  [IN]     Length of the buffer      
 * \return
 *
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 *
 */

void spiflash_read( UINT16 src, UINT32 dst, UINT32 length )
{
    INT32 i;
    UINT8 *psrc, *pdst;

    // Setup command
    spiflashbuf[0] = SPIFLASH_CMD_READ;
    spiflashbuf[1] = ( src >> 16 );
    spiflashbuf[2] = ( src >> 8 );
    spiflashbuf[3] = ( src >> 0 );

    // Execute spiflash read cycle
    spiflash_cycle( spiflashbuf, length + 5 );

    // Copy returned data
    pdst = ( UINT8 * )dst;
    psrc = spiflashbuf + 3;
    for ( i = 0 ; i < length ; i++ )
        *pdst++ = *psrc++;
}
/** **************************************************************************
 * \n \brief Routine to write a page of Spiflash 
 *
 * This routine write a page of data to Spiflash device.This routine writes
 * the data byte by byte till the page size every time it checks whether page  
 * has exceed the pagesize .It takes buffer to be written as a parameter along with 
 * that pagenumber from where data to be read and length of the buffer.
 *
 *
 * \param src     [IN]     Buffer to be written
 * \param dst     [IN]     pagenumber where data to be written
 * \param length  [IN]     Length of the buffer
 *
 *
 */

void spiflash_write( UINT32 src, UINT16 dst, UINT32 length )
{
    INT32 i;
    INT32 bytes_left;
    INT32 bytes_to_program;
    UINT8 *psrc;

    /* Establish source */
    psrc = ( UINT8 * )src;
    bytes_left = length;

    while ( bytes_left > 0 )
    {
        bytes_to_program = bytes_left;

        /* Most to program is SPIFLASH_CMD_BLOCKSIZE */
        if ( bytes_to_program > SPIFLASH_PAGESIZE )
             bytes_to_program = SPIFLASH_PAGESIZE;

        /* Make sure you don't run off the end of a block */
        if ( ( dst & SPIFLASH_PAGEMASK ) != ( ( dst + bytes_to_program ) & SPIFLASH_PAGEMASK ) )
            bytes_to_program -= ( dst + bytes_to_program ) - ( ( dst + bytes_to_program ) & SPIFLASH_PAGEMASK );

        /* Issue WPEN */
        spiflashbuf[0] = SPIFLASH_CMD_WREN;
        spiflash_cycle( spiflashbuf, 1 );

        /* Create command block for program operation */
        spiflashbuf[0] = SPIFLASH_CMD_WRITE;
        spiflashbuf[1] = ( UINT8 )( dst >> 16 );
        spiflashbuf[2] = ( UINT8 )( dst >> 8 );
        spiflashbuf[3] = ( UINT8 )( dst );

        for ( i = 0 ; i < bytes_to_program ; i++ )
            spiflashbuf[3+i] = *psrc++;

        /* Execute write command */
        spiflash_cycle( spiflashbuf, bytes_to_program + 2 );

        /* Wait while busy */
        while( ( spiflash_status( ) & 0x01 ) );

        /* Get ready for next iteration */
        bytes_left -= bytes_to_program;
        dst += bytes_to_program;
    }
}
